<?php

namespace TdySports\Table;

use TdySports\Model;
use TdyCommons\Table\Table;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Delete;
use Zend\Db\Sql\Insert;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Update;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * Class Sports
 *
 * @package TdySports\Table
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Sports extends Table
{

    /**
     * @param string $q
     * @param array  $exclusion
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $exclusion = [], $filters = [], $sort = [], $paginated = false)
    {
        $this->getLogger()->debug($this->indent . 'Sports::fetch() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like($this->columns['slug'], '%' . $q . '%'));
            $where->addPredicate(new Predicate\Like($this->columns['name'], '%' . $q . '%'));
            $conditions[] = $where;
        }

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                if (is_array($v)) {
                    $where->addPredicate(new Predicate\In($k, $v));
                } else {
                    $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
                }
            }
            $conditions[] = $where;
        }

        if (!empty($exclusion)) {
            $where = new Where([]);
            $where->addPredicate(new Predicate\NotIn('id', $exclusion));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (!is_null($sort)) {
            if ($this->isSortEmpty($sort)) {
                $select->order([
                    $this->columns['ordering'] => 'ASC',
                    $this->columns['name']     => 'ASC',
                ]);
            } else {
                $select->order($sort);
            }
        } else {
            $select->order($sort);
        }

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Sport());

        if ($paginated) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: YES');

            $paginatorAdapter = new DbSelect($select, $this->getTableGateway()->getSql(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Paginator instance.');
            $this->getLogger()->debug($this->indent . 'Sports::fetch() end.');

            return $paginator;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: NO');

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);

            $results = $statement->execute();
            $resultSet->initialize($results);

            if (!$resultSet) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns FALSE');
                $this->getLogger()->debug($this->indent . 'Sports::fetch() end.');

                return false;
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns HydratingResultSet instance.');
                $this->getLogger()->debug($this->indent . 'Sports::fetch() end.');

                return $resultSet;
            }
        }
    }

    /**
     * Returns the Sport.
     *
     * @param int $id
     *
     * @return Model\Sport|null
     * @throws \Exception
     */
    public function get($id)
    {
        $this->getLogger()->debug($this->indent . 'Sports::get() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());

        $where = new Where([]);
        $where->addPredicate(new Predicate\Operator($this->columns['id'], Predicate\Operator::OP_EQ, $id));

        $select->where($where);

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Sport());

        $sql       = new Sql($this->tableGateway->getAdapter());
        $statement = $sql->prepareStatementForSqlObject($select);

        $results = $statement->execute();
        $resultSet->initialize($results);

        if (!$resultSet) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns NULL');
            $this->getLogger()->debug($this->indent . 'Sports::get() end.');

            return null;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Model\Sport instance.');
            $this->getLogger()->debug($this->indent . 'Sports::get() end.');

            /** @var Model\Sport $row */
            $row = $resultSet ->current();

            return $row;
        }
    }

    /**
     * @param $slug
     *
     * @return Model\Sport|null
     */
    public function getBySlug($slug)
    {
        $this->getLogger()->debug($this->indent . 'Sports::getBySlug() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());

        $where = new Where([]);
        $where->addPredicate(new Predicate\Operator($this->columns['slug'], Predicate\Operator::OP_EQ, $slug));

        $select->where($where);

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Sport());

        $sql       = new Sql($this->tableGateway->getAdapter());
        $statement = $sql->prepareStatementForSqlObject($select);

        $results = $statement->execute();
        $resultSet->initialize($results);

        if (!$resultSet) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns NULL');
            $this->getLogger()->debug($this->indent . 'Sports::getBySlug() end.');

            return null;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Model\Sport instance.');
            $this->getLogger()->debug($this->indent . 'Sports::getBySlug() end.');

            /** @var Model\Sport $row */
            $row = $resultSet ->current();

            return $row;
        }
    }

    /**
     * @param Model\Sport $sport
     *
     * @return Model\Sport
     * @throws \Exception
     */
    public function save(Model\Sport $sport)
    {
        $this->getLogger()->debug($this->indent . 'Sports::save() start.');

        $data = [
            $this->columns['slug']        => $sport->slug,
            $this->columns['name']        => $sport->name,
            $this->columns['description'] => $sport->description,
        ];

        $id = (int) $sport->id;
        if ($id == 0) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Inserting Sport record.');

            $data['is_active']  = 1;
            $data['created_by'] = $sport->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');

            $insert = new Insert($this->getTableGateway()->getTable());
            $insert->values($data);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $insert->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($insert);

            $statement->execute();

            $id = $this->getAdapter()->getDriver()->getLastGeneratedValue();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ' . $id . ' has been inserted.');
            $this->getLogger()->debug($this->indent . 'Sports::save() end.');

            return $this->get($id);
        } else {
            if ($this->get($id)) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Updating Sport record.');

                $data[$this->columns['updated-by']] = $sport->updatedBy;
                $data[$this->columns['updated-on']] = date('Y-m-d H:i:s');

                $update = new Update($this->getTableGateway()->getTable());
                $update->set($data);
                $update->where([$this->columns['id'] => $id]);

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

                $sql       = new Sql($this->tableGateway->getAdapter());
                $statement = $sql->prepareStatementForSqlObject($update);

                $statement->execute();

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ' . $id . ' has been updated.');
                $this->getLogger()->debug($this->indent . 'Sports::save() end.');

                return $this->get($id);
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ID ' . $id . ' does not exists. Update failed.');
                $this->getLogger()->debug($this->indent . 'Sports::save() end.');

                throw new \Exception('Sport ID does not exist');
            }
        }
    }

    /**
     * @param Model\Sport $sport
     *
     * @throws \Exception
     */
    public function delete(Model\Sport $sport)
    {
        $this->getLogger()->debug($this->indent . 'Sports::delete() start.');

        if ($this->get($sport->id)) {
            $data = [
                $this->columns['is-active']  => 0,
                $this->columns['deleted-by'] => $sport->deletedBy,
                $this->columns['deleted-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $sport->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ' . $sport->id . ' has been deleted.');
            $this->getLogger()->debug($this->indent . 'Sports::delete() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ID ' . $sport->id . ' does not exists. Deletion failed.');
            $this->getLogger()->debug($this->indent . 'Sports::delete() end.');

            throw new \Exception('Sport ID does not exist');
        }
    }

    /**
     * Restore Sport to the table. Sets the Sport to active.
     *
     * @param Model\Sport $sport
     *
     * @throws \Exception
     */
    public function restore(Model\Sport $sport)
    {
        $this->getLogger()->debug($this->indent . 'Sports::restore() start.');

        if ($this->get($sport->id)) {
            $data = [
                $this->columns['is-active']  => 1,
                $this->columns['updated-by'] => $sport->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $sport->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ' . $sport->id . ' has been restored.');
            $this->getLogger()->debug($this->indent . 'Sports::restore() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ID ' . $sport->id . ' does not exists. Restoration failed.');
            $this->getLogger()->debug($this->indent . 'Sports::restore() end.');

            throw new \Exception('Sport ID does not exist');
        }
    }

    /**
     * Permanently delete the Sport.
     *
     * @param Model\Sport $sport
     *
     * @throws \Exception
     */
    public function purge(Model\Sport $sport)
    {
        $this->getLogger()->debug($this->indent . 'Sports::purge() start.');

        if ($this->get($sport->id)) {
            $update = new Delete($this->getTableGateway()->getTable());
            $update->where([$this->columns['id'] => $sport->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ' . $sport->id . ' has been purged, as in permanently deleted.');
            $this->getLogger()->debug($this->indent . 'Sports::purge() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Sport ID ' . $sport->id . ' does not exists. Purge failed.');
            $this->getLogger()->debug($this->indent . 'Sports::purge() end.');

            throw new \Exception('Sport ID does not exist');
        }

    }

}
