<?php

namespace TdyPlayers\Model;

use TdyAvatars\Model\Avatar;
use TdyAvatars\Table\Avatars;
use TdyCommons\Filter\Word\LowercaseFirst;
use TdyCommons\Model\Model;
use Zend\Filter\FilterChain;
use Zend\Filter\Word\DashToCamelCase;
use Zend\Filter\Word\UnderscoreToCamelCase;
use Zend\ServiceManager\ServiceLocatorInterface;
use Zend\ServiceManager\ServiceManager;

/**
 * Class Player
 *
 * @package TdyPlayers\Model
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Player extends Model
{

    /**
     * The unique ID for Player.
     *
     * @var int
     */
    public $id = 0;

    /**
     * The name of the player. This is used for logging in.
     *
     * @var string
     */
    public $username = '';

    /**
     * Player password.
     *
     * @var string
     */
    public $password = '';

    /**
     * Player family name. This is usually father's surname.
     *
     * @var string
     */
    public $lastName = '';

    /**
     * Player given name, or what some call it, first name.
     *
     * @var string
     */
    public $firstName = '';

    /**
     * Player's mother's maiden family name. Many call it middle name.
     *
     * @var string
     */
    public $middleName = '';

    /**
     * Player's suffix. Examples are II, Jr., Sr., III
     *
     * @var string
     */
    public $suffix = '';

    /**
     * Player's nickname.
     *
     * @var string
     */
    public $nickname = '';

    /**
     * Player's email address.
     *
     * @var string
     */
    public $email = '';

    /**
     * @var string
     */
    public $phone = '';

    /**
     * Birthdate of the Player
     *
     * @var null|\DateTime
     */
    public $birthdate = null;

    /**
     * @var string
     */
    public $question1 = '';

    /**
     * @var string
     */
    public $question2 = '';

    /**
     * @var int
     */
    public $avatarId = 1;

    /**
     * Status if the Player is banned.
     *
     * @var int
     */
    public $isBanned = 0;

    /**
     * Date and Time when the Player is banned.
     *
     * @var null|\DateTime
     */
    public $bannedOn = null;

    /**
     * Reason of banning a player.
     *
     * @var null|string
     */
    public $bannedReason = null;

    /**
     * @var null|ServiceManager|ServiceLocatorInterface
     */
    protected $sm;

    /**
     * @param null $data
     * @param null $sm
     */
    public function __construct($data = null, $sm = null)
    {
        if (!empty($data) && is_array($data)) {
            $this->exchangeArray($data);
        }

        $this->sm = $sm;
    }

    /**
     * @param null|ServiceManager|ServiceLocatorInterface $sm
     */
    public function setServiceLocator($sm)
    {
        $this->sm = $sm;
    }

    /**
     * Returns the full name of the Player.
     *
     * <ul>
     *     <li>F - Family name</li>
     *     <li>G - Given name</li>
     *     <li>M - Middle name</li>
     *     <li>S - Suffix. Examples are: II, Jr., Sr., etc.</li>
     * </ul>
     *
     * @param string $format
     *
     * @return string
     */
    public function fullName($format = 'F, G S')
    {
        $name   = '';
        $format = str_split($format);
        foreach ($format as $f) {
            $name .= $this->replaceFormat($f);
        }

        return trim($name);
    }

    /**
     * Replace the character to a name.
     *
     * @param string $char
     *
     * @return string
     */
    protected function replaceFormat($char)
    {
        switch ($char) {
            default:
                $val = $char;
                break;
            case 'F':
                $val = trim($this->lastName);
                break;
            case 'f':
                $val = trim(strtoupper(substr($this->lastName, 0, 1)));
                break;
            case 'G':
                $val = trim($this->firstName);
                break;
            case 'g':
                $token = explode(' ', $this->firstName);
                $init  = '';
                if (count($token) > 0) {
                    foreach ($token as $t) {
                        $init .= strtoupper(substr($t, 0, 1));
                    }
                    $val = trim($init);
                } else {
                    $val = trim($this->firstName);
                }
                break;
            case 'h':
                $val = trim(strtoupper(substr($this->firstName, 0, 1)));
                break;
            case 'J':
                if (!empty($this->suffix)) {
                    $val = trim($this->lastName . ', ' . $this->suffix);
                } else {
                    $val = trim($this->lastName);
                }
                break;
            case 'M':
                $val = trim($this->middleName);
                break;
            case 'm':
                $val = trim(strtoupper(substr($this->middleName, 0, 1)));
                break;
            case 'S':
                $val = trim($this->suffix);
                break;
        }

        return $val;
    }

    /**
     * @see \Application\Model\Model::exchangeArray();
     *
     * @param array $data
     */
    public function exchangeArray($data)
    {
        parent::exchangeArray($data);

        foreach ($data as $key => $value) {
            $filter = new FilterChain();
            if (strpos($key, '-') !== false) {
                $filter->attach(new DashToCamelCase());
            } else {
                $filter->attach(new UnderscoreToCamelCase());
            }

            $filter->attach(new LowercaseFirst());
            $nKey = $filter->filter($key);

            if (!property_exists(__CLASS__, $nKey)) {
                continue;
            } else {
                $inArray = in_array($nKey, ['id']);
                if ($inArray) {
                    $this->$nKey = (int) $value;
                } else {
                    if ($nKey == 'birthdate') {
                        if (!is_null($value)) {
                            $this->$nKey = new \DateTime($value);
                        } else {
                            $this->$nKey = null;
                        }
                    } else {
                        $this->$nKey = $value;
                    }
                }
            }
        }
    }

    /**
     * @return Avatar
     */
    public function getAvatar()
    {
        if (is_null($this->sm)) {
            return new Avatar();
        }

        /**
         * @var Avatars $table
         * @var Avatar  $avatar
         */
        $table  = $this->sm->get('TdyAvatars\Table\Avatars');
        $avatar = $table->get($this->avatarId);

        if (is_null($avatar)) {
            return new Avatar();
        }

        $avatar->setServiceLocator($this->sm);

        return $avatar;
    }

    /**
     * @see parent::toStdClass()
     *
     * @return \StdClass
     */
    public function toStdClass()
    {
        $object = parent::toStdClass();

        $fullName           = new \StdClass;
        $fullName->western  = $this->fullName('G J');
        $fullName->eastern  = $this->fullName('F G S');
        $fullName->clerical = $this->fullName('F, G S');
        $fullName->initials = $this->fullName('hmf');

        $object->fullName = $fullName;

        // Showing passwords in REST is a bad idea, even it is encrypted.
        unset($object->password);

        if (is_null($this->sm)) {
            return $object;
        }

        unset($object->avatarId);
        $object->avatar = $this->getAvatar()->toStdClass();

        return $object;
    }

}
