<?php

namespace TdyGames\Model;

use TdyCommons\Filter\Word\LowercaseFirst;
use TdyCommons\Model\Model;
use TdyGames\Table\Contests;
use TdySports\Model\League;
use TdySports\Model\Sport;
use TdySports\Model\Week;
use TdySports\Table\Leagues;
use TdySports\Table\Sports;
use TdySports\Table\Weeks;
use Zend\Filter\FilterChain;
use Zend\Filter\Word\DashToCamelCase;
use Zend\Filter\Word\UnderscoreToCamelCase;
use Zend\ServiceManager\ServiceLocatorInterface;
use Zend\ServiceManager\ServiceManager;

/**
 * Class Contest
 *
 * @package TdyGames\Model
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Contest extends Model
{

    /**
     * The unique ID for Game.
     *
     * @var int
     */
    public $id = 0;

    /**
     * @var string
     */
    public $slug = '';

    /**
     * @var string
     */
    public $name = '';

    /**
     * @var string
     */
    public $description = '';

    /**
     * @var int
     */
    public $sportId = 0;

    /**
     * @var int
     */
    public $leagueId = 0;

    /**
     * @var int
     */
    public $weekId = 0;

    /**
     * @var int
     */
    public $minPlayers = 0;

    /**
     * @var int
     */
    public $maxPlayers = 0;

    /**
     * @var int
     */
    public $joinedPlayers = 0;

    /**
     * @var float
     */
    public $entranceFee = 0.00;

    /**
     * @var float
     */
    public $prize = 0.00;

    /**
     * @var int
     */
    public $playerId = 0;


    /**
     * @var null|ServiceManager|ServiceLocatorInterface
     */
    protected $sm;

    /**
     * @param null $data
     * @param null $sm
     */
    public function __construct($data = null, $sm = null)
    {
        if (!empty($data) && is_array($data)) {
            $this->exchangeArray($data);
        }

        $this->sm = $sm;
    }

    /**
     * @param null|ServiceManager|ServiceLocatorInterface $sm
     */
    public function setServiceLocator($sm)
    {
        $this->sm = $sm;
    }

    /**
     * @return null|Sport
     */
    public function getSport()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Sports $table
         * @var Sport  $sport
         */
        $table = $this->sm->get('TdySports\Table\Sports');
        $sport = $table->get($this->sportId);

        return $sport;
    }

    /**
     * @return null|League
     */
    public function getLeague()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Leagues $table
         * @var League  $sport
         */
        $table  = $this->sm->get('TdySports\Table\Leagues');
        $league = $table->get($this->leagueId);

        return $league;
    }

    /**
     * @return null|Week
     */
    public function getWeek()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Weeks $table
         * @var Week  $sport
         */
        $table = $this->sm->get('TdySports\Table\Weeks');
        $week  = $table->get($this->weekId);

        return $week;
    }

    /**
     * @see Model::exchangeArray()
     *
     * @param array $data
     */
    public function exchangeArray($data)
    {
        parent::exchangeArray($data);

        foreach ($data as $key => $value) {
            $filter = new FilterChain();
            if (strpos($key, '-') !== false) {
                $filter->attach(new DashToCamelCase());
            } else {
                $filter->attach(new UnderscoreToCamelCase());
            }

            $filter->attach(new LowercaseFirst());
            $nKey = $filter->filter($key);

            if (!property_exists(__CLASS__, $nKey)) {
                continue;
            } else {
                $integers = in_array($nKey, ['id', 'sportId', 'leagueId', 'minPlayers', 'maxPlayers', 'joinedPlayers']);
                if ($integers) {
                    $this->$nKey = (int) $value;
                } else {
                    $this->$nKey = $value;
                }
            }
        }
    }

    protected function isJoined($playerId)
    {
        if (is_null($this->sm)) {
            return false;
        }

        /**
         * @var Contests $table
         * @var bool     $joined
         */
        $table  = $this->sm->get('TdyGames\Table\Contests');
        $joined = $table->isJoined($this->id, $playerId);

        return $joined;
    }

    /**
     * @return \StdClass
     */
    public function toStdClass()
    {
        $object = parent::toStdClass();

        if (!is_null($this->sm)) {
            unset($object->sportId);

            $sport         = $this->getSport();
            $object->sport = $sport->toStdClass();

            unset($object->leagueId);

            $league         = $this->getLeague();
            $object->league = $league->toStdClass();

            unset($object->weekId);

            $week         = $this->getWeek();
            $object->week = $week->toStdClass();

            $isJoined         = $this->isJoined($this->playerId);
            $object->isJoined = $isJoined;

        } else {
            unset($object->playerId);
        }

        return $object;
    }

}
