<?php

namespace TdySports\Table;

use TdySports\Model;
use TdyCommons\Table\Table;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Delete;
use Zend\Db\Sql\Insert;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Update;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * Class Standings
 *
 * @package TdySports\Table
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Standings extends Table
{

    /**
     * @param string $q
     * @param array  $exclusion
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $exclusion = [], $filters = [], $sort = [], $paginated = false)
    {
        $this->getLogger()->debug($this->indent . 'Standings::fetch() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like($this->columns['slug'], '%' . $q . '%'));
            $where->addPredicate(new Predicate\Like($this->columns['name'], '%' . $q . '%'));
            $conditions[] = $where;
        }

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                if (is_array($v)) {
                    $where->addPredicate(new Predicate\In($k, $v));
                } else {
                    $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
                }
            }
            $conditions[] = $where;
        }

        if (!empty($exclusion)) {
            $where = new Where([]);
            $where->addPredicate(new Predicate\NotIn('id', $exclusion));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (!is_null($sort)) {
            if ($this->isSortEmpty($sort)) {
                $select->order(
                    [
                        $this->columns['id'] => 'ASC',
                    ]
                );
            } else {
                $select->order($sort);
            }
        } else {
            $select->order($sort);
        }

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Standing());

        if ($paginated) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: YES');

            $paginatorAdapter = new DbSelect($select, $this->getTableGateway()->getSql(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Paginator instance.');
            $this->getLogger()->debug($this->indent . 'Standings::fetch() end.');

            return $paginator;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: NO');

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);

            $results = $statement->execute();
            $resultSet->initialize($results);

            if (!$resultSet) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns FALSE');
                $this->getLogger()->debug($this->indent . 'Standings::fetch() end.');

                return false;
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns HydratingResultSet instance.');
                $this->getLogger()->debug($this->indent . 'Standings::fetch() end.');

                return $resultSet;
            }
        }
    }

    /**
     * Returns the Standing.
     *
     * @param int $id
     *
     * @return Model\Standing|null
     * @throws \Exception
     */
    public function get($id)
    {
        $rowset = $this->tableGateway->select([$this->columns['id'] => $id]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * Returns the Standing.
     *
     * @param array $params
     *
     * @return Model\Standing|null
     * @throws \Exception
     */
    public function getByParams($params)
    {
        $this->getLogger()->debug($this->indent . 'Standings::getByParams() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $select->where($params);

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\Standing());

        $sql       = new Sql($this->tableGateway->getAdapter());
        $statement = $sql->prepareStatementForSqlObject($select);

        $results = $statement->execute();
        $resultSet->initialize($results);

        if (!$resultSet) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns NULL.');
            $this->getLogger()->debug($this->indent . 'Standings::getByParams() end.');

            return null;
        } else {
            $standings = iterator_to_array($resultSet);

            if ($standings[0] && isset($standings[0])) {

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Standing instance.');
                $this->getLogger()->debug($this->indent . 'Standings::getByParams() end.');

                return $standings[0];
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns NULL.');
                $this->getLogger()->debug($this->indent . 'Standings::getByParams() end.');

                return null;

            }
        }
    }

    /**
     * @param Model\Standing $standing
     *
     * @return Model\Standing
     * @throws \Exception
     */
    public function save(Model\Standing $standing)
    {
        $this->getLogger()->debug($this->indent . 'Standings::save() start.');

        $data = [
            $this->columns['team-id']    => $standing->teamId,
            $this->columns['league-id']  => $standing->leagueId,
            $this->columns['season-id']  => $standing->seasonId,
            $this->columns['wins']       => $standing->wins,
            $this->columns['losses']     => $standing->losses,
            $this->columns['ties']       => $standing->ties,
            $this->columns['percentage'] => $standing->percentage,
        ];

        $id = (int) $standing->id;
        if ($id == 0) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Inserting Standing record.');

            $data['is_active']  = 1;
            $data['created_by'] = $standing->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');

            $insert = new Insert($this->getTableGateway()->getTable());
            $insert->values($data);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $insert->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($insert);

            $statement->execute();

            $id = $this->getAdapter()->getDriver()->getLastGeneratedValue();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ' . $id . ' has been inserted.');
            $this->getLogger()->debug($this->indent . 'Standings::save() end.');

            return $this->get($id);
        } else {
            if ($this->get($id)) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Updating Standing record.');

                $data[$this->columns['updated-by']] = $standing->updatedBy;
                $data[$this->columns['updated-on']] = date('Y-m-d H:i:s');

                $update = new Update($this->getTableGateway()->getTable());
                $update->set($data);
                $update->where([$this->columns['id'] => $id]);

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

                $sql       = new Sql($this->tableGateway->getAdapter());
                $statement = $sql->prepareStatementForSqlObject($update);

                $statement->execute();

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ' . $id . ' has been updated.');
                $this->getLogger()->debug($this->indent . 'Standings::save() end.');

                return $this->get($id);
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ID ' . $id . ' does not exists. Update failed.');
                $this->getLogger()->debug($this->indent . 'Standings::save() end.');

                throw new \Exception('Sport ID does not exist');
            }
        }
    }

    /**
     * @param Model\Standing $standing
     *
     * @throws \Exception
     */
    public function delete(Model\Standing $standing)
    {
        $this->getLogger()->debug($this->indent . 'Standings::delete() start.');

        if ($this->get($standing->id)) {
            $data = [
                $this->columns['is-active']  => 0,
                $this->columns['deleted-by'] => $standing->deletedBy,
                $this->columns['deleted-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $standing->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ' . $standing->id . ' has been deleted.');
            $this->getLogger()->debug($this->indent . 'Standings::delete() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ID ' . $standing->id . ' does not exists. Deletion failed.');
            $this->getLogger()->debug($this->indent . 'Standings::delete() end.');

            throw new \Exception('Sport ID does not exist');
        }
    }

    /**
     * Restore Sport to the table. Sets the Sport to active.
     *
     * @param Model\Standing $standing
     *
     * @throws \Exception
     */
    public function restore(Model\Standing $standing)
    {
        $this->getLogger()->debug($this->indent . 'Standings::restore() start.');

        if ($this->get($standing->id)) {
            $data = [
                $this->columns['is-active']  => 1,
                $this->columns['updated-by'] => $standing->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $standing->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ' . $standing->id . ' has been restored.');
            $this->getLogger()->debug($this->indent . 'Standings::restore() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ID ' . $standing->id . ' does not exists. Restoration failed.');
            $this->getLogger()->debug($this->indent . 'Standings::restore() end.');

            throw new \Exception('Standing ID does not exist');
        }
    }

    /**
     * Permanently delete the Standing.
     *
     * @param Model\Standing $standing
     *
     * @throws \Exception
     */
    public function purge(Model\Standing $standing)
    {
        $this->getLogger()->debug($this->indent . 'Standings::purge() start.');

        if ($this->get($standing->id)) {
            $update = new Delete($this->getTableGateway()->getTable());
            $update->where([$this->columns['id'] => $standing->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ' . $standing->id . ' has been purged, as in permanently deleted.');
            $this->getLogger()->debug($this->indent . 'Standings::purge() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Standing ID ' . $standing->id . ' does not exists. Purge failed.');
            $this->getLogger()->debug($this->indent . 'Standings::purge() end.');

            throw new \Exception('Standing ID does not exist');
        }

    }

}
