<?php

namespace TdySports\Controller\Rest;

use TdySports\Model\Standing;
use TdySports\InputFilter\Standing as StandingInputFilter;
use TdyCommons\Controller\BaseRestController;
use Zend\View\Model\JsonModel;

class StandingsController extends BaseRestController
{
    protected $identifierName = 'id';

    /**
     * @return JsonModel
     */
    public function getList()
    {
        $this->getLogger()->debug('StandingsController::getList() start.');

        $q = $this->params()->fromQuery('q', '');
        $g = $this->params()->fromQuery('g', '');
        $p = (int) $this->params()->fromQuery('p', 1);
        $l = (int) $this->params()->fromQuery('l', $this->getSettingsTable()->get('items-per-page'));
        $s = $this->params()->fromQuery('s', []);
        $f = $this->params()->fromQuery('f', []);
        $x = [];

        $isPaginable = $l > 0;


        $resultSet = $this->getStandingsTable()->setIndent(self::LOGGER_INDENT)->fetch($q, $x, $f, $s, $isPaginable);

        if ($isPaginable) {
            $resultSet->setCurrentPageNumber((int) $p);
            $resultSet->setItemCountPerPage($l);
        }

        if ($p > 0) {
            $i = (($p - 1) * $l);
        } else {
            $i = 0;
        }

        $standings = iterator_to_array($resultSet, false);

        /**
         * @var int      $k
         * @var Standing $standing
         */
        foreach ($standings as $k => $standing) {
            $standing->setServiceLocator($this->getServiceLocator());
            $standings[$k]->row = ++$i;
            $standings[$k]      = $standing->toStdClass();
        }

        if ($g == '1') {
            $grouped = [];

            /** @var \stdClass $standing */
            foreach ($standings as $standing) {
                if (!isset($grouped[$standing->team->group->slug])) {
                    $grouped[$standing->team->group->slug] = [
                        'info' => $standing->team->group,
                        'teams' => []
                    ];
                }

                $grouped[$standing->team->group->slug]['teams'][] = $standing;
            }

            $standings = $grouped;
        }


        $this->getLogger()->debug('StandingsController::getList() end.');

        return $this->statusOk([
            'entries'    => $standings,
            'pagination' => [
                'pageNumber' => $p,
                'pageSize'   => $l,
                'totalItems' => $isPaginable ? $resultSet->getTotalItemCount() : 1,
                'totalPages' => $isPaginable ? $resultSet->count() : 1,
            ],

        ]);
    }

    /**
     * @param int|string $id
     *
     * @return JsonModel
     */
    public function get($id)
    {
        $standing = $this->getStandingsTable()->get($id);
        $standing->setServiceLocator($this->getServiceLocator());

        return $this->statusOk([
            'entry'      => $standing->toStdClass(),
            'pagination' => [
                'pageNumber' => 1,
                'pageSize'   => 1,
                'totalItems' => 1,
                'totalPages' => 1,
            ],
        ]);
    }

    /**
     * @param mixed $data
     *
     * @return JsonModel
     */
    public function create($data)
    {
        $this->getLogger()->debug('StandingsController::create() start.');

        $data['id'] = 0;

        $inputFilter = new StandingInputFilter($this->getServiceLocator());
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $data = $inputFilter->getValues();

            $standing            = new Standing($data);
            $standing->createdBy = $this->getCurrentPlayer()->id;

            $standing = $this->getStandingsTable()->setIndent(self::LOGGER_INDENT)->save($standing);

            $standing->setServiceLocator($this->getServiceLocator());

            $this->getLogger()->info(self::LOGGER_INDENT . 'Standing ' . $standing->name . ' has been created.');
            $this->getLogger()->debug('StandingsController::create() end.');

            return $this->statusOk([
                'entry'      => $standing->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Bugging out.');
            $this->getLogger()->debug('StandingsController::create() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages()
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     * @param mixed      $data
     *
     * @return JsonModel
     */
    public function update($id, $data)
    {
        $this->getLogger()->debug('StandingsController::update() start.');

        $data['id'] = (int) $id;

        $inputFilter = new StandingInputFilter($this->getServiceLocator(), true);
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $standing            = new Standing($data);
            $standing->updatedBy = $this->getCurrentPlayer()->id;
            $standing            = $this->getStandingsTable()->setIndent(self::LOGGER_INDENT)->save($standing);

            $standing->setServiceLocator($this->getServiceLocator());

            $this->getLogger()->info(self::LOGGER_INDENT . 'Standing ' . $standing->name . ' has been updated.');
            $this->getLogger()->debug('StandingsController::update() end.');

            return $this->statusOk([
                'entry'      => $standing->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Will not update.');
            $this->getLogger()->debug('StandingsController::update() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages(),
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     *
     * @return JsonModel
     * @throws \Exception
     */
    public function delete($id)
    {
        $this->getLogger()->debug('StandingsController::delete() start.');

        $this->getLogger()->debug(self::LOGGER_INDENT . 'Getting info.');

        if ($id > 0) {
            $standing = $this->getStandingsTable()->get($id);
            if ($standing) {

                $this->getLogger()->debug(self::LOGGER_INDENT . 'Standing ID  : ' . $id);
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Standing Name: ' . $standing->name);

                $standing->deletedBy = $this->getCurrentPlayer()->id;
                $this->getStandingsTable()->setIndent(self::LOGGER_INDENT)->delete($standing);

                $this->getLogger()->info(self::LOGGER_INDENT . 'Standing ' . $standing->name . ' has been deleted.');
                $this->getLogger()->debug('StandingsController::delete() end.');

                return $this->statusOk([
                    'entry'      => null,
                    'pagination' => [
                        'pageNumber' => 1,
                        'pageSize'   => 1,
                        'totalItems' => 1,
                        'totalPages' => 1,
                    ],
                ]);
            } else {
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Standing ID ' . $id . ' does not exists.');
                $this->getLogger()->debug('StandingsController::delete() end.');

                return $this->statusNotFound();
            }
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Standing ID is requred for deletion.');
            $this->getLogger()->debug('StandingsController::delete() end.');

            return $this->statusNotFound();
        }
    }

}
