<?php

namespace TdySports\Controller\Rest;

use TdySports\Model\Schedule;
use TdySports\InputFilter\Schedule as ScheduleInputFilter;
use TdyCommons\Controller\BaseRestController;
use Zend\View\Model\JsonModel;

class SchedulesController extends BaseRestController
{
    protected $identifierName = 'id';

    /**
     * @return JsonModel
     */
    public function getList()
    {
        $this->getLogger()->debug('SchedulesController::getList() start.');

        $q = $this->params()->fromQuery('q', '');
        $p = (int) $this->params()->fromQuery('p', 1);
        $l = (int) $this->params()->fromQuery('l', $this->getSettingsTable()->get('items-per-page'));
        $s = $this->params()->fromQuery('s', []);
        $f = $this->params()->fromQuery('f', []);
        $x = [];

        $isPaginable = $l > 0;

        $resultSet = $this->getSchedulesTable()->setIndent(self::LOGGER_INDENT)->fetch($q, $x, $f, $s, $isPaginable);

        if ($isPaginable) {
            $resultSet->setCurrentPageNumber((int)$p);
            $resultSet->setItemCountPerPage($l);
        }

        if ($p > 0) {
            $i = (($p - 1) * $l);
        } else {
            $i = 0;
        }

        $schedules = iterator_to_array($resultSet, false);

        /**
         * @var int      $k
         * @var Schedule $schedule
         */
        foreach ($schedules as $k => $schedule) {
            $schedule->setServiceLocator($this->getServiceLocator());
            $schedules[$k]->row = ++$i;
            $schedules[$k]      = $schedule->toStdClass();
        }

        $this->getLogger()->debug('SchedulesController::getList() end.');

        return $this->statusOk([
            'entries'    => $schedules,
            'pagination' => [
                'pageNumber' => $isPaginable ? $p : 1,
                'pageSize'   => $isPaginable ? $l : count($schedules),
                'totalItems' => $isPaginable ? $resultSet->getTotalItemCount() : count($schedules),
                'totalPages' => $isPaginable ? $resultSet->count() : 1,
            ],

        ]);
    }

    /**
     * @param int|string $id
     *
     * @return JsonModel
     */
    public function get($id)
    {
        $schedule = $this->getSchedulesTable()->get($id);
        $schedule->setServiceLocator($this->getServiceLocator());

        return $this->statusOk([
            'entry'      => $schedule->toStdClass(),
            'pagination' => [
                'pageNumber' => 1,
                'pageSize'   => 1,
                'totalItems' => 1,
                'totalPages' => 1,
            ],
        ]);
    }

    /**
     * @param mixed $data
     *
     * @return JsonModel
     */
    public function create($data)
    {
        $this->getLogger()->debug('SchedulesController::create() start.');

        $data['id'] = 0;

        $inputFilter = new ScheduleInputFilter($this->getServiceLocator());
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $data = $inputFilter->getValues();

            $schedule            = new Schedule($data);
            $schedule->createdBy = $this->getCurrentPlayer()->id;

            $schedule = $this->getSchedulesTable()->setIndent(self::LOGGER_INDENT)->save($schedule);

            $schedule->setServiceLocator($this->getServiceLocator());

            $this->getLogger()->info(self::LOGGER_INDENT . 'Schedule ' . $schedule->name . ' has been created.');
            $this->getLogger()->debug('SchedulesController::create() end.');

            return $this->statusOk([
                'entry'      => $schedule->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Bugging out.');
            $this->getLogger()->debug('SchedulesController::create() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages()
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     * @param mixed      $data
     *
     * @return JsonModel
     */
    public function update($id, $data)
    {
        $this->getLogger()->debug('SchedulesController::update() start.');

        $data['id'] = (int) $id;

        $inputFilter = new ScheduleInputFilter($this->getServiceLocator(), true);
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is valid. Continue.');

            $schedule            = new Schedule($data);
            $schedule->updatedBy = $this->getCurrentPlayer()->id;
            $schedule            = $this->getSchedulesTable()->setIndent(self::LOGGER_INDENT)->save($schedule);

            $schedule->setServiceLocator($this->getServiceLocator());

            $this->getLogger()->info(self::LOGGER_INDENT . 'Schedule ' . $schedule->name . ' has been updated.');
            $this->getLogger()->debug('SchedulesController::update() end.');

            return $this->statusOk([
                'entry'      => $schedule->toStdClass(),
                'pagination' => [
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ],
            ]);
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Input Filter is invalid. Will not update.');
            $this->getLogger()->debug('SchedulesController::update() end.');

            return $this->statusBadRequest([
                'system'    => [
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ],
                'companies' => [
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages(),
                ]
            ]);
        }
    }

    /**
     * @param int|string $id
     *
     * @return JsonModel
     * @throws \Exception
     */
    public function delete($id)
    {
        $this->getLogger()->debug('SchedulesController::delete() start.');

        $this->getLogger()->debug(self::LOGGER_INDENT . 'Getting info.');

        if ($id > 0) {
            $schedule = $this->getSchedulesTable()->get($id);
            if ($schedule) {

                $this->getLogger()->debug(self::LOGGER_INDENT . 'Schedule ID  : ' . $id);
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Schedule Name: ' . $schedule->name);

                $schedule->deletedBy = $this->getCurrentPlayer()->id;
                $this->getSchedulesTable()->setIndent(self::LOGGER_INDENT)->delete($schedule);

                $this->getLogger()->info(self::LOGGER_INDENT . 'Schedule ' . $schedule->name . ' has been deleted.');
                $this->getLogger()->debug('SchedulesController::delete() end.');

                return $this->statusOk([
                    'entry'      => null,
                    'pagination' => [
                        'pageNumber' => 1,
                        'pageSize'   => 1,
                        'totalItems' => 1,
                        'totalPages' => 1,
                    ],
                ]);
            } else {
                $this->getLogger()->debug(self::LOGGER_INDENT . 'Schedule ID ' . $id . ' does not exists.');
                $this->getLogger()->debug('SchedulesController::delete() end.');

                return $this->statusNotFound();
            }
        } else {
            $this->getLogger()->debug(self::LOGGER_INDENT . 'Schedule ID is requred for deletion.');
            $this->getLogger()->debug('SchedulesController::delete() end.');

            return $this->statusNotFound();
        }
    }

}
