<?php

namespace TdyCommons\Controller;

use TdyCommons\Model\Option;
use TdyCommons\InputFilter\Option as OptionInputFilter;
use Zend\Filter\File\RenameUpload;
use Zend\View\Model\JsonModel;

/**
 * Class OptionsController
 *
 * @package TdyCommons\Controller
 */
class OptionsController extends BaseRestController
{
    /**
     * @see BaseApiController::identifierName
     *
     * @var string
     */
    protected $identifierName = 'opt';

    /**
     * @api      {get} /selections/:sel/options List Options
     * @apiName  Get Options
     * @apiGroup Options
     *
     * @apiParam {String} [q=] Keyword to search.
     * @apiParam {Number} [p=1] Page number.
     * @apiParam {Number} [l=20] Number of options to display at a time.
     * @apiparam {Array}  [s=] Field sorting. Not yet implemented.
     *
     * @apiSuccess (200 OK) {Object}   httpStatus HTTP status code.
     * @apiSuccess (200 OK) {Object[]} data    List of Options.
     *
     * @return JsonModel
     */
    public function getList()
    {
        $c = $this->params()->fromRoute('sel', 0);
        $q = $this->params()->fromQuery('q', '');
        $p = (int) $this->params()->fromQuery('p', 1);
        $l = (int) $this->params()->fromQuery('l', $this->getSettingsTable()->get('items-per-page'));
        $s = $this->params()->fromQuery('s', array());

        $resultSet = $this->getOptionsTable()->fetch($c, $q, $s, true);
        $resultSet->setCurrentPageNumber((int) $p);
        $resultSet->setItemCountPerPage($l);

        if ($p > 0) {
            $i = (($p - 1) * $l);
        } else {
            $i = 0;
        }

        $options = iterator_to_array($resultSet);

        /**
         * @var int    $k
         * @var Option $option
         */
        foreach ($options as $k => $option) {
            $option->setServiceLocator($this->getServiceLocator());
            $option->rowNo = ++ $i;
            $options[ $k ] = $option->toStdClass();
        }

        return $this->statusOk(array(
            'entries'    => $options,
            'pagination' => array(
                'pageNumber' => $p,
                'pageSize'   => $l,
                'totalItems' => $resultSet->getTotalItemCount(),
                'totalPages' => $resultSet->count(),
            ),
        ));
    }

    /**
     * @api      {get} /selections/:sel/options/:opt Get Option
     * @apiName  Get Options
     * @apiGroup Options
     *
     * @apiParam {Number} id The Option ID.
     *
     * @apiSuccess (200 OK) {Object}  httpStatus HTTP status code.
     * @apiSuccess (200 OK) {Object}  data    Returns the Option.
     *
     * @param int $opt
     *
     * @return JsonModel
     */
    public function get($opt)
    {
        $option = $this->getOptionsTable()->get($opt);
        $option->setServiceLocator($this->getServiceLocator());

        return $this->statusOk(array(
            'entry'      => $option->toStdClass(),
            'pagination' => array(
                'pageNumber' => 1,
                'pageSize'   => 1,
                'totalItems' => 1,
                'totalPages' => 1,
            ),
        ));
    }

    /**
     * @api        {post} /selections/:sel/options Create New Option
     * @apiVersion 0.0.1
     * @apiName    Post Option
     * @apiGroup   Options
     *
     * @apiSuccess (200 OK) {Object}  httpStatus HTTP status code.
     * @apiSuccess (200 OK) {Object}  data    Returns the Option object.
     *
     * @param array $data
     *
     * @return JsonModel
     */
    public function create($data)
    {
        $sel              = $this->params()->fromRoute('sel', 0);
        $httpRequest      = $this->getRequest();
        $contentTypeValue = $httpRequest->getHeaders()->get('Content-Type')->getFieldValue();

        list($contentType) = explode(';', trim($contentTypeValue));

        switch ($contentType) {
            case 'multipart/form-data':
                $file   = $httpRequest->getFiles('file');
                $filter = new RenameUpload(array(
                    'target'    => APP_ROOT_PATH . "/app/data/uploads/sheet.upd",
                    'randomize' => true,
                ));

                $file = $filter->filter($file);
                $path = $file['tmp_name'];

                $valid  = false;
                $reader = null;
                $types  = array('Excel2007', 'Excel5', 'CSV');
                foreach ($types as $type) {
                    $reader = \PHPExcel_IOFactory::createReader($type);
                    if ($reader->canRead($path)) {
                        $valid = true;
                        break;
                    }
                }

                if ($valid) {
                    /** @var \PHPExcel $objPHPExcel */
                    $objPHPExcel = $reader->load($path);
                    $worksheet   = $objPHPExcel->getActiveSheet();

                    $i       = 1;
                    $j       = 1;
                    $options = array();

                    foreach ($worksheet->getRowIterator() as $row) {
                        $cellIterator = $row->getCellIterator();
                        $cellIterator->setIterateOnlyExistingCells(false);

                        if ($i == 1 && $worksheet->getCell('A1')->getValue() == 'slug') {
                            $i ++;
                            continue;
                        }

                        $option              = new Option();
                        $option->selectionId = $sel;
                        $option->slug        = $worksheet->getCell('A' . $i)->getValue();
                        $option->name        = $worksheet->getCell('B' . $i)->getValue();

                        $newObject        = $this->getOptionsTable()->save($option);
                        $newObject->rowNo = $j;
                        $options[]        = $newObject->toStdClass();

                        $i ++;
                        $j ++;
                    }

                    unlink($path);

                    return $this->statusOk(array(
                        'entries'    => $options,
                        'pagination' => array(
                            'pageNumber' => 1,
                            'pageSize'   => $i,
                            'totalItems' => $i,
                            'totalPages' => 1,
                        ),
                    ));

                } else {
                    return $this->statusBadRequest(array(
                        'system'     => array(
                            'id'          => 4002,
                            'description' => 'Invalid Import File'
                        ),
                        'selections' => array(
                            'description' => 'The file for import is invalid. Only accepts files in CSV, XLS, or XLSX format.',
                        )
                    ));
                }

                break;
            case 'application/json':
            case 'text/json':
            default:
                $inputFilter = new OptionInputFilter();
                $inputFilter->setData($data);
                if ($inputFilter->isValid()) {
                    $option              = new Option($data);
                    $option->selectionId = $sel;
                    $option->createdBy   = $this->getCurrentUser()->id;
                    $option              = $this->getOptionsTable()->save($option);

                    return $this->statusOk(array(
                        'entry'      => $option->toStdClass(),
                        'pagination' => array(
                            'pageNumber' => 1,
                            'pageSize'   => 1,
                            'totalItems' => 1,
                            'totalPages' => 1,
                        ),
                    ));
                } else {
                    return $this->statusBadRequest(array(
                        'system'  => array(
                            'id'          => 4001,
                            'description' => 'Invalid Request'
                        ),
                        'options' => array(
                            'description' => 'Submission is invalid.',
                            'details'     => $inputFilter->getMessages()
                        )
                    ));
                }
                break;
        }

    }

    /**
     * @api        {put} /selections/:sel/options/:opt Update Option
     * @apiVersion 0.0.1
     * @apiName    Put Option
     * @apiGroup   Options
     *
     * @apiSuccess (200 OK) {Object}  httpStatus HTTP status code.
     * @apiSuccess (200 OK) {Object}  data    Returns the Option object.
     *
     * @param int   $opt
     * @param array $data
     *
     * @return JsonModel
     */
    public function update($opt, $data)
    {
        $inputFilter = new OptionInputFilter();
        $inputFilter->setData($data);

        if ($inputFilter->isValid()) {
            $option              = new Option($data);
            $option->selectionId = $this->params()->fromRoute('sel', 0);
            $option->updatedBy   = $this->getCurrentUser()->id;
            $option              = $this->getOptionsTable()->save($option);

            return $this->statusOk(array(
                'entry'      => $option->toStdClass(),
                'pagination' => array(
                    'pageNumber' => 1,
                    'pageSize'   => 1,
                    'totalItems' => 1,
                    'totalPages' => 1,
                ),
            ));
        } else {
            return $this->statusBadRequest(array(
                'system'  => array(
                    'id'          => 4001,
                    'description' => 'Invalid Request'
                ),
                'options' => array(
                    'description' => 'Submission is invalid.',
                    'details'     => $inputFilter->getMessages()
                )
            ));
        }
    }

    /**
     * @api        {delete} /selections/:sel/options/:opt Delete Option
     * @apiVersion 0.0.1
     * @apiName    Delete Option
     * @apiGroup   Options
     *
     * @apiSuccess (200 OK) {Object}  httpStatus HTTP status code.
     * @apiSuccess (200 OK) null      data
     *
     * @param int $opt
     *
     * @return JsonModel
     */
    public function delete($opt)
    {
        if ($opt > 0) {
            $option = $this->getOptionsTable()->get($opt);
            if ($option) {
                $option->deletedBy = $this->getCurrentUser()->id;
                $this->getOptionsTable()->delete($option);

                return $this->statusOk(array(
                    'entry'      => null,
                    'pagination' => array(
                        'pageNumber' => 1,
                        'pageSize'   => 1,
                        'totalItems' => 1,
                        'totalPages' => 1,
                    ),
                ));
            } else {
                return $this->statusNotFound();
            }
        } else {
            return $this->statusNotFound();
        }
    }

}
