<?php

namespace TdySports\Model;

use TdyCommons\Filter\Word\LowercaseFirst;
use TdyCommons\Model\Model;
use TdySports\Table\Leagues;
use TdySports\Table\Seasons;
use TdySports\Table\Sports;
use TdySports\Table\Statuses;
use TdySports\Table\Teams;
use TdySports\Table\Weeks;
use Zend\Filter\FilterChain;
use Zend\Filter\Word\DashToCamelCase;
use Zend\Filter\Word\UnderscoreToCamelCase;
use Zend\ServiceManager\ServiceLocatorInterface;
use Zend\ServiceManager\ServiceManager;

/**
 * Class Group
 *
 * @package TdySports\Model
 *
 * @author  James Lloyd Atwil <james@nadows.com>
 */
class Schedule extends Model
{
    const STATUS_UPCOMING = 0;
    const STATUS_CANCELLED = 1;
    const STATUS_ONGOING = 8;
    const STATUS_FINISHED = 9;

    /**
     * The unique ID for Sport.
     *
     * @var int
     */
    public $id = 0;

    /**
     * @var int
     */
    public $sportId = 0;

    /**
     * @var int
     */
    public $leagueId = 0;

    /**
     * @var int
     */
    public $seasonId = 0;

    /**
     * @var int
     */
    public $weekId = 0;

    /**
     * @var null|\DateTime
     */
    public $playDateTime = null;

    /**
     * @var string
     */
    public $slug = '';

    /**
     * @var string
     */
    public $name = '';

    /**
     * @var string
     */
    public $description = '';

    /**
     * @var int
     */
    public $teamHome = 0;

    /**
     * @var int
     */
    public $teamAway = 0;

    /**
     * @var string
     */
    public $venue = '';

    /**
     * @var int
     */
    public $status = self::STATUS_UPCOMING;

    /**
     * @var int
     */
    public $scoreHome = 0;

    /**
     * @var int
     */
    public $scoreAway = 0;

    /**
     * @var int
     */
    public $ending = 0;

    /**
     * @var
     */
    public $winner;

    /**
     * @var
     */
    public $loser;

    /**
     * @var null|ServiceManager|ServiceLocatorInterface
     */
    protected $sm;

    /**
     * @param null $data
     * @param null $sm
     */
    public function __construct($data = null, $sm = null)
    {
        if (!empty($data) && is_array($data)) {
            $this->exchangeArray($data);
        }

        $this->sm = $sm;
    }

    /**
     * @param null|ServiceManager|ServiceLocatorInterface $sm
     */
    public function setServiceLocator($sm)
    {
        $this->sm = $sm;
    }

    /**
     * @see Model::exchangeArray()
     *
     * @param array $data
     */
    public function exchangeArray($data)
    {
        parent::exchangeArray($data);

        foreach ($data as $key => $value) {
            $filter = new FilterChain();
            if (strpos($key, '-') !== false) {
                $filter->attach(new DashToCamelCase());
            } else {
                $filter->attach(new UnderscoreToCamelCase());
            }

            $filter->attach(new LowercaseFirst());
            $nKey = $filter->filter($key);

            if (!property_exists(__CLASS__, $nKey)) {
                continue;
            } else {
                $inArray = in_array($nKey, ['id', 'sportId', 'leagueId', 'weekId', 'scoreHome', 'scoreAway', 'ordering']);
                if ($inArray) {
                    $this->$nKey = (int) $value;
                } else {
                    $this->$nKey = $value;
                }
            }
        }
    }


    /**
     * @return null|Sport
     */
    public function getSport()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Sports $table
         * @var Sport  $sport
         */
        $table = $this->sm->get('TdySports\Table\Sports');
        $sport = $table->get($this->sportId);

        return $sport;
    }

    /**
     * @return null|League
     */
    public function getLeague()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Leagues $table
         * @var League  $sport
         */
        $table  = $this->sm->get('TdySports\Table\Leagues');
        $league = $table->get($this->leagueId);

        return $league;
    }

    /**
     * @return null|Season
     */
    public function getSeason()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Seasons $table
         * @var Season  $sport
         */
        $table  = $this->sm->get('TdySports\Table\Seasons');
        $season = $table->get($this->seasonId);

        return $season;
    }

    /**
     * @return null|Week
     */
    public function getWeek()
    {
        if (is_null($this->sm)) {
            return null;
        }

        /**
         * @var Weeks $table
         * @var Week  $sport
         */
        $table = $this->sm->get('TdySports\Table\Weeks');
        $week  = $table->get($this->weekId);

        return $week;
    }

    /**
     * @param int $id
     *
     * @return null|Team
     */
    public function getTeam($id)
    {
        if (is_null($this->sm)) {
            return null;
        }

        if ($id > 0) {
            /**
             * @var Teams $table
             * @var Team  $sport
             */
            $table = $this->sm->get('TdySports\Table\Teams');
            $team  = $table->get($id);

            return $team;
        } else {
            return new Team(null);
        }
    }

    /**
     * @param int $id
     *
     * @return null|Status
     */
    public function getStatus($id)
    {
        if (is_null($this->sm)) {
            return null;
        }

        /** @var Statuses $table */
        $table  = $this->sm->get('TdySports\Table\Statuses');
        $status = $table->get($id);

        return $status;
    }

    /**
     * @return \StdClass
     */
    public function toStdClass()
    {
        $object = parent::toStdClass();

        if (is_null($this->sm)) {
            return $object;
        }

        unset($object->sportId);

        $sport         = $this->getSport();
        $object->sport = $sport->toStdClass();

        unset($object->leagueId);

        $league         = $this->getLeague();
        $object->league = $league->toStdClass();

        unset($object->seasonId);

        $season         = $this->getSeason();
        $object->season = $season->toStdClass();

        unset($object->weekId);

        $week         = $this->getWeek();
        $object->week = $week->toStdClass();

        unset($object->teamHome);

        $homeTeam         = $this->getTeam($this->teamHome);
        $object->homeTeam = $homeTeam->toStdClass();
        $object->teamHome = $homeTeam->toStdClass();

        unset($object->teamAway);

        $awayTeam         = $this->getTeam($this->teamAway);
        $object->awayTeam = $awayTeam->toStdClass();
        $object->teamAway = $awayTeam->toStdClass();

        unset($object->status);

        $status         = $this->getStatus($this->status);
        $object->status = $status->toStdClass();

        return $object;
    }

}
