<?php

namespace TdyCommons\Table;

use TdyCommons\Model\Setting;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Where;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

class Settings extends Table
{

    /**
     * @param string $q
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $filters = [], $sort = [], $paginated = false)
    {
        $select     = new Select($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
            }
            $conditions[] = $where;
        }

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like('[key]', '%' . $q . '%'));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (empty($sort)) {
            $select->order(['id' => 'ASC']);
        } else {
            $select->order($sort);
        }

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Setting());

        if ($paginated) {
            $paginatorAdapter = new DbSelect($select, $this->getAdapter(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            return $paginator;
        } else {
            $sql       = new Sql($this->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);
            $results   = $statement->execute();

            $resultSet->initialize($results);

            if (!$resultSet) {
                return false;
            } else {
                return $resultSet;
            }
        }
    }

    /**
     * @param string|int $id
     * @param bool       $returnObject
     *
     * @return Setting|bool|string
     */
    public function get($id, $returnObject = false)
    {
        $rowset = $this->getTableGateway()->select(function (Select $select) use ($id) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Operator($this->columns['id'], Predicate\Operator::OP_EQ, (int) $id));
            $where->addPredicate(new Predicate\Operator($this->columns['key'], Predicate\Operator::OP_EQ, $id));
            $select->where($where);
        });

        $row = $rowset->current();

        if (!$row) {
            return false;
        } else {
            if ($returnObject) {
                return $row;
            } else {
                return $row->value;
            }
        }
    }

    /**
     * Save the Setting to the table.
     *
     * @param Setting $setting
     *
     * @return Setting
     * @throws \Exception
     */
    public function save(Setting $setting)
    {
        $data = [
            'key' => $setting->key,
            'value' => $setting->value,
        ];

        $id = $setting->id;
        if ($id === 0) {
            $data['is_active']  = 1;
            $data['created_by'] = $setting->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');
            $this->tableGateway->insert($data);
            $id = $this->getLastInsertId();

            return $this->get($id);
        } else {
            if ($this->get($setting->id)) {
                $data['updated_by'] = $setting->updatedBy;
                $data['updated_on'] = date('Y-m-d H:i:s');
                $this->tableGateway->update($data, ['id' => $setting->id]);

                return $this->get($id);
            } else {
                throw new \Exception('Setting ID does not exist');
            }
        }
    }

    /**
     * Delete the Setting from the table. Actually, it simply sets the Setting to inactive.
     *
     * @param Setting $setting
     *
     * @throws \Exception
     */
    public function delete(Setting $setting)
    {
        if ($this->get($setting->id)) {
            $data = [
                'is_active' => 0,
                'deleted_by' => $setting->deletedBy,
                'deleted_on' => date('Y-m-d H:i:s'),
            ];
            $this->tableGateway->update($data, ['id' => $setting->id]);
        } else {
            throw new \Exception('Setting ID does not exist');
        }
    }

    /**
     * Sets the Setting to active.
     *
     * @param Setting $setting
     *
     * @throws \Exception
     */
    public function restore(Setting $setting)
    {
        if ($this->get($setting->id)) {
            $data = [
                'is_active' => 1,
                'updated_by' => $setting->updatedBy,
                'updated_on' => date('Y-m-d H:i:s'),
            ];
            $this->tableGateway->update($data, ['id' => $setting->id]);
        } else {
            throw new \Exception('Setting ID does not exist');
        }
    }

    /**
     * @param Setting $setting
     */
    public function burn(Setting $setting)
    {
        $this->tableGateway->delete(['id' => $setting->id]);
    }

}
