<?php

namespace TdyCommons\Table;

use TdyCommons\Model;
use Zend\Db\ResultSet\HydratingResultSet;
use Zend\Db\Sql\Delete;
use Zend\Db\Sql\Insert;
use Zend\Db\Sql\Sql;
use Zend\Db\Sql\Predicate;
use Zend\Db\Sql\Select;
use Zend\Db\Sql\Update;
use Zend\Db\Sql\Where;
use Zend\Filter\FilterChain;
use Zend\Filter\Word\DashToUnderscore;
use Zend\Paginator\Adapter\DbSelect;
use Zend\Paginator\Paginator;

/**
 * DropdownItems table class.
 *
 * @author James Lloyd Atwil <james@nadows.com>
 */
class DropdownItems extends Table
{

    /**
     * @param string $q
     * @param array  $exclusion
     * @param array  $filters
     * @param array  $sort
     * @param bool   $paginated
     *
     * @return bool|HydratingResultSet|Paginator
     */
    public function fetch($q = '', $exclusion = [], $filters = [], $sort = [], $paginated = false)
    {
        $this->getLogger()->debug($this->indent . 'DropdownItems::fetch() start.');

        $select = new Select();
        $select->from($this->getTableGateway()->getTable());
        $conditions = [];

        $where = new Where([]);
        $where->addPredicate(new Predicate\Expression('is_active = ?', 1));
        $conditions[] = $where;

        if (!empty($q)) {
            $where = new Where([], Predicate\PredicateSet::OP_OR);
            $where->addPredicate(new Predicate\Like($this->columns['slug'], '%' . $q . '%'));
            $where->addPredicate(new Predicate\Like($this->columns['name'], '%' . $q . '%'));
            $conditions[] = $where;
        }

        if (!empty($filters)) {
            $where = new Where([]);
            foreach ($filters as $k => $v) {
                $chain = new FilterChain();
                $chain->attach(new DashToUnderscore());

                $k = $chain->filter($k);

                if (is_array($v)) {
                    $where->addPredicate(new Predicate\In($k, $v));
                } else {
                    $where->addPredicate(new Predicate\Expression($k . ' = ?', $v));
                }
            }
            $conditions[] = $where;
        }

        if (!empty($exclusion)) {
            $where = new Where([]);
            $where->addPredicate(new Predicate\NotIn('id', $exclusion));
            $conditions[] = $where;
        }

        $select->where($conditions, Predicate\PredicateSet::OP_AND);

        if (!is_null($sort)) {
            if ($this->isSortEmpty($sort)) {
                $select->order([
                    $this->columns['ordering'] => 'ASC',
                    $this->columns['name']     => 'ASC',
                ]);
            } else {
                $select->order($sort);
            }
        } else {
            $select->order($sort);
        }

        $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $select->getSqlString($this->getAdapter()->getPlatform()));

        $resultSet = new HydratingResultSet();
        $resultSet->setObjectPrototype(new Model\DropdownItem());

        if ($paginated) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: YES');

            $paginatorAdapter = new DbSelect($select, $this->getTableGateway()->getSql(), $resultSet);
            $paginator        = new Paginator($paginatorAdapter);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns Paginator instance.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::fetch() end.');

            return $paginator;
        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Is Paginable: NO');

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($select);

            $results = $statement->execute();
            $resultSet->initialize($results);

            if (!$resultSet) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns FALSE');
                $this->getLogger()->debug($this->indent . 'DropdownItems::fetch() end.');

                return false;
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Returns HydratingResultSet instance.');
                $this->getLogger()->debug($this->indent . 'DropdownItems::fetch() end.');

                return $resultSet;
            }
        }
    }

    /**
     * Returns the DropdownItem.
     *
     * @param int $id
     *
     * @return Model\DropdownItem|null
     * @throws \Exception
     */
    public function get($id)
    {
        $rowset = $this->tableGateway->select([$this->columns['id'] => $id]);
        $row    = $rowset->current();
        if (!$row) {
            return null;
        }

        return $row;
    }

    /**
     * @param Model\DropdownItem $item
     *
     * @return Model\DropdownItem
     * @throws \Exception
     */
    public function save(Model\DropdownItem $item)
    {
        $this->getLogger()->debug($this->indent . 'DropdownItems::save() start.');

        $data = [
            $this->columns['slug']        => $item->slug,
            $this->columns['name']        => $item->name,
            $this->columns['description'] => $item->description,
            $this->columns['value']       => $item->value,
        ];

        $id = (int) $item->id;
        if ($id == 0) {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Inserting DropdownItem record.');

            $data['is_active']  = 1;
            $data['created_by'] = $item->createdBy;
            $data['created_on'] = date('Y-m-d H:i:s');

            $insert = new Insert($this->getTableGateway()->getTable());
            $insert->values($data);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $insert->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($insert);

            $statement->execute();

            $id = $this->getAdapter()->getDriver()->getLastGeneratedValue();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ' . $id . ' has been inserted.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::save() end.');

            return $this->get($id);
        } else {
            if ($this->get($id)) {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'Updating DropdownItem record.');

                $data[$this->columns['updated-by']] = $item->updatedBy;
                $data[$this->columns['updated-on']] = date('Y-m-d H:i:s');

                $update = new Update($this->getTableGateway()->getTable());
                $update->set($data);
                $update->where([$this->columns['id'] => $id]);

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

                $sql       = new Sql($this->tableGateway->getAdapter());
                $statement = $sql->prepareStatementForSqlObject($update);

                $statement->execute();

                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ' . $id . ' has been updated.');
                $this->getLogger()->debug($this->indent . 'DropdownItems::save() end.');

                return $this->get($id);
            } else {
                $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ID ' . $id . ' does not exists. Update failed.');
                $this->getLogger()->debug($this->indent . 'DropdownItems::save() end.');

                throw new \Exception('DropdownItem ID does not exist');
            }
        }
    }

    /**
     * @param Model\DropdownItem $item
     *
     * @throws \Exception
     */
    public function delete(Model\DropdownItem $item)
    {
        $this->getLogger()->debug($this->indent . 'DropdownItems::delete() start.');

        if ($this->get($item->id)) {
            $data = [
                $this->columns['is-active']  => 0,
                $this->columns['deleted-by'] => $item->deletedBy,
                $this->columns['deleted-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $item->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ' . $item->id . ' has been deleted.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::delete() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ID ' . $item->id . ' does not exists. Deletion failed.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::delete() end.');

            throw new \Exception('DropdownItem ID does not exist');
        }
    }

    /**
     * Restore DropdownItem to the table. Sets the DropdownItem to active.
     *
     * @param Model\DropdownItem $item
     *
     * @throws \Exception
     */
    public function restore(Model\DropdownItem $item)
    {
        $this->getLogger()->debug($this->indent . 'DropdownItems::restore() start.');

        if ($this->get($item->id)) {
            $data = [
                $this->columns['is-active']  => 1,
                $this->columns['updated-by'] => $item->updatedBy,
                $this->columns['updated-on'] => date('Y-m-d H:i:s'),
            ];

            $update = new Update($this->getTableGateway()->getTable());
            $update->set($data);
            $update->where([$this->columns['id'] => $item->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ' . $item->id . ' has been restored.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::restore() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ID ' . $item->id . ' does not exists. Restoration failed.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::restore() end.');

            throw new \Exception('DropdownItem ID does not exist');
        }
    }

    /**
     * Permanently delete the DropdownItem.
     *
     * @param Model\DropdownItem $item
     *
     * @throws \Exception
     */
    public function purge(Model\DropdownItem $item)
    {
        $this->getLogger()->debug($this->indent . 'DropdownItems::purge() start.');

        if ($this->get($item->id)) {
            $update = new Delete($this->getTableGateway()->getTable());
            $update->where([$this->columns['id'] => $item->id]);

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'SQL Statement: ' . $update->getSqlString($this->getAdapter()->getPlatform()));

            $sql       = new Sql($this->tableGateway->getAdapter());
            $statement = $sql->prepareStatementForSqlObject($update);

            $statement->execute();

            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ' . $item->id . ' has been purged, as in permanently deleted.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::purge() end.');

        } else {
            $this->getLogger()->debug($this->indent . self::LOGGER_INDENT . 'DropdownItem ID ' . $item->id . ' does not exists. Purge failed.');
            $this->getLogger()->debug($this->indent . 'DropdownItems::purge() end.');

            throw new \Exception('DropdownItem ID does not exist');
        }

    }

}
